
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 25 10:17:10 2007
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief A brief description of this module 
   *
   *  \version CRYS_ATP_TST_HASHFLOW.c#1:csrc:1
   *  \author Annag
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */
/************* Include Files ****************/
#include "CRYS.h"

#include "CRYS_ATP_TST_Defines.h"
#include "CRYS_ATP_TST_UTIL_Funcs.h"
#include "CRYS_ATP_TST_UTIL_OTF_Funcs.h"
#include "CRYS_ATP_TST_HASHFLOW_Acceptance.h"
#include "CRYS_ATP_TST_AESFLOW_Acceptance.h"
#include "CRYS_ATP_TST_DESFLOW_Acceptance.h"
#include "CRYS_ATP_TST_HASHFLOW_Acceptance_test_vectors.h"
#include "CRYS_ATP_TST_FLOW_Defines.h"

#include "CRYS_INIT_TST.h"
#include "CRYS_FLOW.h"
#include "CRYS_FLOW_error.h"
#include "gen.h"

#include "DX_VOS_Mem.h"
#include "FLOW_error.h"

#if TST_USE_FLOW_DATA

/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

extern DxUint8_t 	                      TST_Input_Data[TST_FLOW_DATA_VEC_SIZE_IN_BYTS];
extern DxUint64_t	                      TST_FLOW_InputData[TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

extern DxUint64_t						  TST_FLOW_OutputData[TST_FLOW_NUMBER_OF_SUPPORTED_ENGINES][TST_FLOW_DATA_VEC_SIZE_IN_BYTS/8];

/* input data size */
extern DxUint32_t                         TST_FLOW_InputDataSize;

extern CRYS_FLOW_Params_t                 TST_FLOWParamsInfo;

extern TST_FLOW_TestCommonDataStruct      TST_TestData;

/* hash engine data */
CRYS_FLOW_HASHEngine_Data_t               TST_FLOWHASHEngineData;

/* output data of HASH/AES_MAC result */
DxUint32_t                                TST_FLOW_Hash_OutputData[CRYS_HASH_RESULT_SIZE_IN_WORDS];

                                    
extern DxBool_t TST_FLOW_IsFinishSizeZero;                                    


/************* Private function prototype ***********************/

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_InitHASHEngineData
 *  
 * Inputs:
 *  TST_index - test index
 *  None
 *
 * Outputs:                                 
 *
 * Description:                                                                 
 *  This function initializes engine data for HASH FLOW operation
 * Algorithm:     
 *	This function initializes HASH flow engine data struct with values from the test vector.
 ******************************************************************************/                                
void CRYS_ATP_TST_FLOW_InitHASHEngineData ( int TST_index ) 
{
	TST_FLOWHASHEngineData.opMode = TST_HASHFLOW_Acceptance_TestData[TST_index].TST_OperationMode;
}

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_InitializeHASHFlowInfo
 *  
 * Inputs:
 *  TST_index - test index
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *	This function initialize HASH flow info parameter.
 *
 * Algorithm:     
 *	1. Initialize Flow info parameter.
 *  2. Init HASH flow
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_InitHASHFlow ( int                 TST_index,
                                        CRYS_FLOW_Type_t    flowType,
                                        char               *TST_MasterTestName, 
                                        DxUint32_t         *TST_HASHFlowId,
                                        CRYS_FLOW_Params_t *pHASHFlowParamsInfo ) 
{
    /********* TST Variables Definitions ************/
    CRYSError_t  TST_TestStatus = TESTPASS;
    
    TST_TestData.inputBigEndianOrderFlag  = TST_HASHFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
	TST_TestData.inputBuffersArraySize    = TST_HASHFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	TST_TestData.inputInterruptId         = TST_HASHFLOW_Acceptance_TestData[TST_index].inputInterruptId;
	TST_TestData.outputBigEndianOrderFlag = TST_HASHFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
	TST_TestData.outputBuffersArraySize   = TST_HASHFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
	TST_TestData.outputInterruptId        = TST_HASHFLOW_Acceptance_TestData[TST_index].outputInterruptId;
	TST_TestData.wordsBuffersOrderFlag    = TST_HASHFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;
	
    /* Initializing Flow info parameter */
 	/************************************/
    
    /* initialize flow init parameters */
 	CRYS_ATP_TST_FLOW_InitParams(TST_index, DX_TRUE, &TST_TestData, pHASHFlowParamsInfo, CRYS_FLOW_HASH);
 	    
 
 	pHASHFlowParamsInfo->outputBuffersArrays = DX_NULL;

	/* initialize HASH engine data */
	CRYS_ATP_TST_FLOW_InitHASHEngineData(TST_index);
	

	/* Init HASH flow */
	/******************/
	TST_ATP_Error = CRYS_FLOW_HASHInit(flowType,
			                           pHASHFlowParamsInfo,
			                           &TST_FLOWHASHEngineData,  
			                           TST_HASHFlowId);
			                           
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_HASHInit is done\n"));

	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_HASHInit",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  			                           
        
    
    return TST_TestStatus;
}


/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_CheckHASHOperation
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function checks some HASH operation
* Algorithm:     
* 1. Execute some HASH operation 
* 2. Compare result to expected one
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_CheckHASHOperation () 
{
	/********* TST Variables Definitions ************/
	/* use test index = 0 */
	DxUint32_t   TST_index = 0;
	CRYSError_t  TST_TestStatus = TESTPASS;
	
	/* Run CRYS_HASH function */
	/**************************/
	TST_ATP_Error = CRYS_HASH(TST_HASHFLOW_Acceptance_TestData[0].TST_OperationMode,
                			  TST_Input_Data,
                			  TST_FLOW_InputDataSize,
                			  TST_FLOW_Hash_OutputData);
        			  
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_HASH is done\n"));
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_HASH",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif          			 
		
	/* compare HASH result with one expected */
	/****************************************/
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
		TST_FLOW_Hash_OutputData,
		TST_HASHFLOW_Acceptance_TestData[0].TST_Expected_output_data,
		CRYS_HASH_RESULT_SIZE_IN_WORDS*4);
    
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CheckHASHOperation is done\n"));
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CheckHASHOperation",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name);

    #if CRYS_ATP_TST_EXIT_ON_ERROR
        if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
    #endif  
	return TST_ATP_Error;

}

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FinishHASHFlow
 *  
 * Inputs:
 *  TST_index          - test index
 *  TST_HASHFlowId     - flow id
 *  TST_MasterTestName - test name
 *  blockSize          - block size for finish
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *	This function finishes HASH flow and check results.
 *
 * Algorithm:     
 *	1. Finish Flow
 *  2. Get results
 *  3. Compare flow output with expected output
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FinishHASHFlow ( int                TST_index, 
                                         DxUint32_t          TST_HASHFlowId, 
                                         char               *TST_MasterTestName,
                                         DxUint32_t          blockSize,
                                         CRYS_FLOW_Params_t *pHASHFlowParamsInfo ) 
{
    /********* TST Variables Definitions ************/
    CRYSError_t  TST_TestStatus = TESTPASS;
    
    
    /* Finish flow */
    /***************/
    TST_ATP_Error = CRYS_FLOW_Finish(TST_HASHFlowId, blockSize);  

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Finish",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

	
	/* Get results */
	/***************/
	do
    {
        
        TST_ATP_Error = CRYS_FLOW_GetResults(TST_HASHFlowId,
		                                 (DxUint8_t*)TST_FLOW_Hash_OutputData,
		                                 CRYS_HASH_RESULT_SIZE_IN_WORDS*4);
	
    }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);
	
		                        
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));                             
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  


	/* Compare flow output with expected output */
	/********************************************/
	/* compare and prints results of test */
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
		(DxUint8_t*)TST_FLOW_Hash_OutputData,
		TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data,
		TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize);


	/*Print the outputted and then the expected result after mem compare */
	ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
		"\nData outputted by engine [output buffer]-- \n",
			(DxUint8_t*)TST_FLOW_Hash_OutputData, 
		    TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize); 

	ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
		"\nData Expected by engine --\n",
		TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data,
		TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize);                 	                                     			
	

	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

    return TST_TestStatus;
    
}

/************************ Public Functions ******************************/

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HASHS_ONETEST_Acceptance
 *  
 * Inputs:
 *  TST_index          - test index
 *  TST_MasterTestName - test name
 *  flowType           - flow type (static or dynamic)
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static HASH FLOW API's
 
 * Algorithm:     
 *  1. Initialize HASH flow with type and other parameters
 *  2. Send interrupts that input and output buffers are ready
 *  3. Finish HASH flow and compare result to expected results
 *  4. Terminate flow
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HASH_ONETEST_Acceptance ( int              TST_index, 
                                                        char             *TST_MasterTestName,
                                                        CRYS_FLOW_Type_t flowType ) 
{

	/********* TST Variables Definitions ************/
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	int                          blockSize = 0;
	/* flow id */
	DxUint32_t                   flowId = 0;
	int                          i = 0;

	
	/************************************************/
	/*********** Start of test **********************/
	/************************************************/

	/* input block size */
	blockSize       = TST_FLOW_InputDataSize/TST_HASHFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;

	/* initialize input buffer: */
	DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
	
	/* Initialize HASH flow parameters and init HASH flow by calling function */
	/* CRYS_FLOW_HASHInit                                                     */
	/**************************************************************************/
	TST_TestStatus = CRYS_ATP_TST_InitHASHFlow(TST_index, flowType, TST_MasterTestName, &flowId, &TST_FLOWParamsInfo);
                                        

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  
    
    /* Send interrupt that input/output buffers are ready */
    /******************************************************/
    	    
    /* Send interrupt that input buffers are ready */
    if (TST_FLOW_IsFinishSizeZero == DX_TRUE)
    {
        for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize; ++i)
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
            
        blockSize = 0;	            
    }
    else
    {
        for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 1; ++i)
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_FLOWParamsInfo.inputInterruptId);
    }
	    
    
    /* Finish HASH flow, get results and compare results with expected  */
    /* by calling functions CRYS_FLOW_Finish and CRYS_FLOW_GetResults   */
    /********************************************************************/
	TST_TestStatus = CRYS_ATP_TST_FinishHASHFlow(TST_index, flowId, TST_MasterTestName, 
 	                                             blockSize, &TST_FLOWParamsInfo);
 	                                             
    #if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
    #endif  

    /* Terminate flow */
    /******************/
	TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
	

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

    return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance*/ 


/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static HASH FLOW API's
 
 * Algorithm:     
 *	This function goes through all tests vectors and execute HASH static flow acceptance test
 *  for each vector
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance () 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance"; 
	
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* go through all tests vectors and execute HASHs flow acceptance test for each vector */
    /*************************************************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HASHFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_HASH_ONETEST_Acceptance(TST_index, TST_MasterTestName, CRYS_FLOW_TypeStatic);
		

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for dynamic HASH FLOW API's
 
 * Algorithm:     
 *	This function goes through all tests vectors and execute HASH dynamic flow acceptance test
 *  for each vector
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance () 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance"; 
	
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* go through all tests vectors and execute HASHs flow acceptance test for each vector */
    /*************************************************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HASHFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_HASH_ONETEST_Acceptance(TST_index, TST_MasterTestName, CRYS_FLOW_TypeDynamic);
		

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance*/ 


/*****************************************************************************
* Function Name:                                                           
*  CRYS_ATP_TST_FLOW_HASHDYNAMIC_AddBuffers_Acceptance
*  
* Inputs:
*  None
*
* Outputs: 
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for dynamic HASH FLOW API's

* Algorithm:     
*  1. Initialize HASH flow with dynamic type and other parameters
*  2. Init HASH flow
*  3. Send interrupts that input and output buffers are ready
*  4. Add buffers to dynamic flow
*  5. Send interrupts that added input and output buffers are ready
*  6. Finish HASH flow
*  7. Get results
*  8. Compare result to expected one
******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HASHDYNAMIC_AddBuffers_Acceptance () 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HASHDYNAMIC_AddBuffers_Acceptance"; 
	
	/* flow id */
	DxUint32_t                   flowId = 0;
	DxUint32_t                   j = 0; 
	DxUint32_t                   inputBuffersArraySize = 1;

	CRYS_FLOW_BufferData_t       inputBuffersArray_ptr[10];
    DxUint16_t                   blockSize;
    
	

	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HASHDYNAMIC_AddBuffers_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/************************************************/
	/*********** Start of test **********************/
	/************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HASHFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		/* Initialize HASH flow with dynamic type and other parameters */
        /**************************************************************/
		TST_TestData.inputBigEndianOrderFlag  = TST_HASHFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
		TST_TestData.inputBuffersArraySize    = TST_HASHFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
		TST_TestData.inputInterruptId         = TST_HASHFLOW_Acceptance_TestData[TST_index].inputInterruptId;
		TST_TestData.outputBigEndianOrderFlag = TST_HASHFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
		TST_TestData.outputBuffersArraySize   = TST_HASHFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
		TST_TestData.outputInterruptId        = TST_HASHFLOW_Acceptance_TestData[TST_index].outputInterruptId;
		TST_TestData.wordsBuffersOrderFlag    = TST_HASHFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;

		blockSize       = TST_FLOW_InputDataSize/TST_HASHFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	

		DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
		
		/* initialize flow init parameters */
		CRYS_ATP_TST_FLOW_InitParams(TST_index, DX_FALSE, &TST_TestData, &TST_FLOWParamsInfo, CRYS_FLOW_HASH);

		/* initialize HASH engine data */
		CRYS_ATP_TST_FLOW_InitHASHEngineData(TST_index);

		/* Init HASH flow */
		/******************/
		TST_ATP_Error = CRYS_FLOW_HASHInit(CRYS_FLOW_TypeDynamic,
			                               &TST_FLOWParamsInfo,
			                               &TST_FLOWHASHEngineData,  
			                               &flowId);

	
		ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_HASHInit is done\n"));

		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_HASHInit",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	    /* Send interrupts that input and output buffers are ready */
        /***********************************************************/
	    
		/* Send interrupt that input buffers are ready */
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);

		/* initialize buffers to add to flow */
		inputBuffersArraySize  = TST_HASHFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
		
        /* prepare buffers to add */
	    for (j = 0; j < inputBuffersArraySize - 1; ++j )
		{
			/* point to the next block of the input buffer */
			inputBuffersArray_ptr[j].buffer_ptr = (DxUint8_t*)TST_FLOW_InputData + (j+1)*blockSize;

			inputBuffersArray_ptr[j].bufferSize = blockSize;                
		}

		/* Add buffers to dynamic flow */
        /*******************************/
		TST_ATP_Error = CRYS_FLOW_AddBuffers(flowId,
                				             inputBuffersArray_ptr,
                				             inputBuffersArraySize - 1,
                				             DX_NULL,
                				             0);
                				             
        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_AddBuffers is done\n"));

		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_AddBuffers",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif                  				             
				
    
        /* Send interrupts that added input and output buffers are ready */
        /*****************************************************************/
            
        /* Send interrupt that input buffers are ready */
        for (j = 0; j < inputBuffersArraySize - 1; ++j )
            CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
                
        blockSize = 0;    
       

		/* Finish HASH flow, get results and compare results with expected  */
        /* by calling functions CRYS_FLOW_Finish and CRYS_FLOW_GetResults  */
        /*******************************************************************/
		TST_TestStatus = CRYS_ATP_TST_FinishHASHFlow(TST_index, flowId, TST_MasterTestName, 
 	                                                 blockSize, &TST_FLOWParamsInfo);
 	                                             
        #if CRYS_ATP_TST_EXIT_ON_ERROR
    	if (TST_TestStatus != TESTPASS) 
    		return TESTFAIL;
        #endif  
    
        /* terminate flow */
        /******************/
        TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
		
		ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n")); 
		
		TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	}/* End of TST_index */     

	
	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HASHDYNAMIC_AddBuffers_Acceptance*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HASH_ONETEST_Acceptance_WithAdditionalEngines
 *  
 * Inputs:
 *  TST_index          - test index
 *  TST_MasterTestName - test name
 *  flowType           - flow type (static or dynamic)
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static HASH FLOW API's
 
 * Algorithm:     
 *  1. Init HASH flow with type and other parameters
 *  2. Send several interrupts that input buffer is ready
 *  3. Execute AES operation and compare with expected result
 *  4. Send interrupt that input buffer is ready, 
 *  5. Execute some DES operation and compare the result of the DES to expected
 *  6. Send interrupts that input buffers are ready
 *  7. Finish HASH flow, get result and compare with expected result 
 *  8.Terminate flow
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HASH_ONETEST_Acceptance_WithAdditionalEngines (
                                                                int              TST_index, 
                                                                char             *TST_MasterTestName,
                                                                CRYS_FLOW_Type_t flowType ) 
{

	/********* TST Variables Definitions ************/
    CRYSError_t             	 TST_TestStatus = TESTPASS;
	/* flow id */
	DxUint32_t                   flowId = 0;
	DxUint16_t                   blockSize = 0;
	int                          i = 0;
	
	
	/************************************************/
	/*********** Start of test **********************/
	/************************************************/
    TST_TestData.inputBigEndianOrderFlag  = TST_HASHFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
	TST_TestData.inputBuffersArraySize    = TST_HASHFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	TST_TestData.inputInterruptId         = TST_HASHFLOW_Acceptance_TestData[TST_index].inputInterruptId;
	TST_TestData.outputBigEndianOrderFlag = TST_HASHFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
	TST_TestData.outputBuffersArraySize   = TST_HASHFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
	TST_TestData.outputInterruptId        = TST_HASHFLOW_Acceptance_TestData[TST_index].outputInterruptId;
	TST_TestData.wordsBuffersOrderFlag    = TST_HASHFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;

	blockSize       = TST_FLOW_InputDataSize/TST_HASHFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	

	/* copy input to aligned input buffer: */
	DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
	
	/* Initialize HASH flow parameters and init HASH flow by calling function */
    /* CRYS_FLOW_HASHInit                                                     */
	/**************************************************************************/
	TST_TestStatus = CRYS_ATP_TST_InitHASHFlow(TST_index, flowType, TST_MasterTestName, &flowId, &TST_FLOWParamsInfo);
	
	#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
    #endif  

	/* Send interrupt that input buffer is ready */
	/***********************************************************/

	/* Send interrupt that input buffers are ready */
	CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	
	/* execute AES operation and compare with expected result */
	/*********************************************************/
	TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckAESOperation();
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"AES operation",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  
	
	/* Send interrupt that input buffers are ready */
	/***********************************************/
	CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	
	/* execute HASH operation and compare with expected result */
	/***********************************************************/
	TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckHASHOperation();
	

	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"HASH operation",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

    /* Send interrupt that input buffer is ready */
    /*********************************************/
	CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	
    /* execute DES operation and compare with expected result */
	/***********************************************************/
    TST_ATP_Error = CRYS_ATP_TST_FLOW_CheckDESOperation();
    
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"DES operation",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

    /* Send interrupts that input buffers are ready */
    /*********************************************/
    
    /* Send interrupt that input buffers are ready */
    if ((TST_FLOW_IsFinishSizeZero == DX_TRUE) ||
        (TST_FLOWParamsInfo.inputBuffersArraySize < 4))
    {
        /* inputBuffersArraySize should be minimum 3 */
        for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 3; ++i)
	    {
	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	    }
	    
	    blockSize = 0;
    }
    else
    {
        for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize - 4; ++i)
	    {
	        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	    }
    }
	

	/* Finish HASH flow, get results and compare results with expected  */
    /* by calling functions CRYS_FLOW_Finish and CRYS_FLOW_GetResults  */
    /*******************************************************************/
	TST_TestStatus = CRYS_ATP_TST_FinishHASHFlow(TST_index, flowId, TST_MasterTestName, 
 	                                             blockSize, &TST_FLOWParamsInfo);
 	                                             
    #if CRYS_ATP_TST_EXIT_ON_ERROR
    if (TST_TestStatus != TESTPASS) 
        return TESTFAIL;
    #endif  
    
    
    /* Terminate flow */
    /******************/
    TST_ATP_Error = CRYS_FLOW_Terminate(flowId);
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HASH_ONETEST_Acceptance_WithAdditionalEngines*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance_WithAdditionalEngines
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static HASH FLOW API's
 
 * Algorithm:     
 *  This function goes through all flow vectors and execute acceptance test with 
 *  additional enginesfor each vector
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance_WithAdditionalEngines ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance_WithAdditionalEngines"; 
	
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance_WithAdditionalEngines Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* go through all flow vectors and execute acceptance test */
	/***********************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HASHFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
	    TST_TestStatus = CRYS_ATP_TST_FLOW_HASH_ONETEST_Acceptance_WithAdditionalEngines(
	                                                                    TST_index,
	                                                                    TST_MasterTestName,
	                                                                    CRYS_FLOW_TypeStatic);
	                                                                    
#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HASHSTATIC_Acceptance_WithAdditionalEngines*/ 


/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance_WithAdditionalEngines
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for dynamic HASH FLOW API's
 * Algorithm:     
 *  This function goes through all flow vectors and execute acceptance test with 
 *  additional enginesfor each vector
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance_WithAdditionalEngines ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance_WithAdditionalEngines"; 
	
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance_WithAdditionalEngines Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* go through all flow vectors and execute acceptance test */
	/***********************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HASHFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
	    TST_TestStatus = CRYS_ATP_TST_FLOW_HASH_ONETEST_Acceptance_WithAdditionalEngines(
	                                                                    TST_index,
	                                                                    TST_MasterTestName,
	                                                                    CRYS_FLOW_TypeDynamic);
	                                                                    
#if CRYS_ATP_TST_EXIT_ON_ERROR
		if (TST_TestStatus != TESTPASS) 
			return TESTFAIL;
#endif  

	}/* End of TST_index */     

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HASHDYNAMIC_Acceptance_WithAdditionalEngines*/ 

/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HASHSTATIC_ONETEST_Cyclic_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static HASH FLOW API's
 
 * Algorithm:     
 * 1. Initialize HASH flow with static type and other parameters 
 * 2. Init HASH flow
 * 3. Send interrupts to execute cyclic flow operation 
 * 4. Finish HASH flow
 * 5. Get results
 * 6. Terminate flow
 * 7. Compare output data to expected result
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HASHSTATIC_ONETEST_Cyclic_Acceptance ( int   TST_index,
                                                                     char *TST_MasterTestName ) 
{

	/********* TST Variables Definitions ************/
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	
	/* flow id */
	DxUint32_t                   flowId = 0;
	DxUint32_t                   i = 0;

	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;

	
	TST_TestData.inputBigEndianOrderFlag  = TST_HASHFLOW_Acceptance_TestData[TST_index].inputBigEndianOrderFlag;
	TST_TestData.inputBuffersArraySize    = TST_HASHFLOW_Acceptance_TestData[TST_index].inputBuffersArraySize;
	TST_TestData.inputInterruptId         = TST_HASHFLOW_Acceptance_TestData[TST_index].inputInterruptId;
	TST_TestData.outputBigEndianOrderFlag = TST_HASHFLOW_Acceptance_TestData[TST_index].outputBigEndianOrderFlag;
	TST_TestData.outputBuffersArraySize   = TST_HASHFLOW_Acceptance_TestData[TST_index].outputBuffersArraySize;
	TST_TestData.outputInterruptId        = TST_HASHFLOW_Acceptance_TestData[TST_index].outputInterruptId;
	TST_TestData.wordsBuffersOrderFlag    = TST_HASHFLOW_Acceptance_TestData[TST_index].wordsBuffersOrderFlag;

	/* prepare input buffer: */
	DX_VOS_FastMemCpy((DxUint8_t*)TST_FLOW_InputData, TST_Input_Data, TST_FLOW_InputDataSize);
	
	/* Initialize HASH flow data */
    /****************************/
    
	/* initialize flow init parameters */
	CRYS_ATP_TST_FLOW_InitParams(TST_index, DX_TRUE, &TST_TestData, &TST_FLOWParamsInfo, CRYS_FLOW_HASH);
	
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;

	/* initialize HASH engine data */
	CRYS_ATP_TST_FLOW_InitHASHEngineData(TST_index);


	/* Init HASH flow */
	/******************/
	TST_ATP_Error = CRYS_FLOW_HASHInit(CRYS_FLOW_TypeStatic,
			&TST_FLOWParamsInfo,
			&TST_FLOWHASHEngineData,  
			&flowId);

	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_HASHInit is done\n"));

	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_HASHInit",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  
    
    /* Send interrupts to execute cyclic flow operation */
    /****************************************************/
    
	/* Send interrupt that input buffers are ready */
	for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize; ++i)
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	
	
    /* Send interrupt that input buffers are ready  second time */
    for (i = 0; i < TST_FLOWParamsInfo.inputBuffersArraySize; ++i)
        CRYS_ATP_TST_FLOW_SendInterrupt(TST_TestData.inputInterruptId);
	
	
    /* Finish HASH flow */
	/********************/
	/* finish size should be 0 in cyclic buffers flow */
	CRYS_FLOW_Finish(flowId,0);  

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Finish is done\n"));                             

	
	/* Get results */
	/***************/
	do
    {
        
        TST_ATP_Error = CRYS_FLOW_GetResults(flowId,
		                                 (DxUint8_t*)TST_FLOW_Hash_OutputData,
		                                 CRYS_HASH_RESULT_SIZE_IN_WORDS*4);
	
    }while(TST_ATP_Error == DX_SEP_FLOW_GET_RESULT_NOT_READY_ERROR);
	

	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_GetResults is done\n"));
	
	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_GetResults",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  
	
	/* Terminate flow */
    /******************/
	CRYS_FLOW_Terminate(flowId);
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"CRYS_FLOW_Terminate is done\n"));                             
        
    TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,"CRYS_FLOW_Terminate",TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);
 	
    #if CRYS_ATP_TST_EXIT_ON_ERROR
    if (TST_TestStatus != TESTPASS) 
        return TESTFAIL;
    #endif  

	/* Compare output data to expected result */
	/********************************************************************/ 
	
	
	/* compare and prints results of test */
	TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(
		(DxUint8_t*)TST_FLOW_Hash_OutputData,
		TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Expected_cyclicOperation_data,
	    TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize);


	/*Print the outputted and then the expected result after mem compare */
	ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
		"\nData outputted by engine [output buffer]-- \n",
			(DxUint8_t*)TST_FLOW_Hash_OutputData, 
		    TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize); 

	ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
		"\nData Expected by engine --\n",
		TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Expected_output_data,
		TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Expected_dataSize);                 	                                     			
	

	TST_TestStatus = TST_ATP_CheckErrorReturn(TST_ATP_Error,TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_HASHFLOW_Acceptance_TestData[TST_index].TST_Name,TST_MasterTestName);

#if CRYS_ATP_TST_EXIT_ON_ERROR
	if (TST_TestStatus != TESTPASS) 
		return TESTFAIL;
#endif  

	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HASHSTATIC_ONETEST_Acceptance_WithAdditionalEngines*/ 


/*****************************************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_FLOW_HASHSTATIC_Cyclic_Acceptance
 *  
 * Inputs:
 *  None
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for static HASH FLOW API's
 
 * Algorithm:     
 *	This function initializes HASH flow with static type and other parameters,
 * and executes several times flow on the same input buffer.It goes throw all test
 * vectors and check cyclic acceptance.
 ******************************************************************************/                                
CRYSError_t CRYS_ATP_TST_FLOW_HASHSTATIC_Cyclic_Acceptance ( void ) 
{

	/********* TST Variables Definitions ************/
	int                     	 TST_index;
	CRYSError_t             	 TST_TestStatus = TESTPASS;
	char                     	 TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_FLOW_HASHSTATIC_Cyclic_Acceptance"; 
	
	
	/* initialize input and output buffers */
	TST_FLOWParamsInfo.inputBuffersArrays  = DX_NULL;
	TST_FLOWParamsInfo.outputBuffersArrays = DX_NULL;
	
	
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n"));
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_FLOW_HASHSTATIC_Cyclic_Acceptance Tests ********************* \n"));      
	ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************** \n \n"));


	/* This function goes throw all test vectors and check cyclic acceptance */
	/*************************************************************************/
	for (TST_index =0 ; TST_index < TST_NumberOf_HASHFLOW_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
	{
		TST_TestStatus = CRYS_ATP_TST_FLOW_HASHSTATIC_ONETEST_Cyclic_Acceptance(TST_index,
		                                                                        TST_MasterTestName);
		                                                                        
#if CRYS_ATP_TST_EXIT_ON_ERROR
	    if (TST_TestStatus != TESTPASS) 
            return TESTFAIL;
#endif  		                                                               
		                                                                        
		
	}/* End of TST_index */  
	   
	return TST_TestStatus;
}/* End of CRYS_ATP_TST_FLOW_HASHSTATIC_Cyclic_Acceptance*/ 

#endif
